/**
* Script: state.js
* Written by: Radnen
* Updated: 12/15/2010
**/

/**
* Global State Handler:
*  - Control's state objects.
**/
var StateManager = ({
	states: [],
	showCursor: true, // toggle this to show or hide cursor.
	preRender: null,  // special use, to go behind everything.
	
	/**
	* push();
	*  - Array wrapper for state stack, adds a state object.
	*  - state: the state object to add onto the state-stack.
	**/
	push: function(state) {
		if (this.states.length > 0) {
			this.states[this.states.length-1].active = false;
			this.states[this.states.length-1].onLeave.execute();
		}
		this.states.push(state);
		state.onEnter.execute();
	},
	
	/**
	* pop();
	*  - Array wrapper for state stack, removes and returns a state object.
	**/
	pop: function() {
		var state = this.states.pop();
		state.onLeave.execute();
		if (this.states.length > 0) {
			this.states[this.states.length-1].active = true;
			this.states[this.states.length-1].onEnter.execute();
		}
		return state;
	},
	
	/**
	* update();
	*  - Updates only the current top-most state.
	**/
	update: function() {
		if (this.states[this.states.length-1]) this.states[this.states.length-1].update.execute();
	},
	
	/**
	* render();
	*  - Renders all states, bottom to top. Top state will always be drawn physically on top.
	**/
	render: function() {
		if (this.preRender && this.states.length > 0) this.preRender();
		Utility.foreach(this.states, function(state) {
			state.render.execute();
		});
		if (this.showCursor) Cursor.draw();
	},
	
	/**
	* execute();
	*  - updates and renders the states.
	**/
	execute: function() {
		while(this.states.length > 0) {
			this.update();
			this.render();
			FlipScreen();
		}
	},
	
	/**
	* purgeStates();
	*  - Clears all of the states; like quitting the game.
	**/
	purgeStates: function() {
		this.states = [];
	},
	
	/**
	* findState(text);
	*  - Finds and returns a state by name
	*  - name: the name of the state to find and return.
	**/
	findState: function(name) {
		var found = foreach(this.states, function (state) {
			if (state.name == name) return state.name;
		});
		if (!found) return null;
		else return found;
	},
	
	/**
	* getStateList();
	*  - returns a list of state name strings, not a list of the states.
	*    this is useful for debug purposes.
	**/
	getStateList: function() {
		var arr = [];
		Utility.foreach(this.states, function(state) { arr.push(state.name); });
		return arr;
	},
	
	/**
	* removeState(state);
	*  - Removes the state from the state stack.
	*  - state: the state object to remove or, the name of state
	*           object to find and remove.
	**/
	removeState: function(state) {
		if (typeof state != "string") state = state.name;
		Utility.remove(this.states, state, "name");
	}
});

/**
* State object:
*  - name: name of the state for debug and ease of use.
**/
function State(name) {
	/* Event Handlers: */
	this.update  = new Event(this);
	this.render  = new Event(this);
	this.onEnter = new Event(this);
	this.onLeave = new Event(this);
	
	/* Triggered Events: */
	this.doInput   = function(key){}; // remember to add a key argument when overloading.
	this.preRender = function(){};
	
	/* Internal State Variables */
	this.active = true;
	this.name = name;
	
	this.update.add(function() {
		while(AreKeysLeft()) {
			this.doInput(GetKey());
		}		
	});

	this.render.add(function() {
		this.preRender();
	});
	
	/**
	* show();
	*  - Use this to alternatively display the state.
	**/
	this.show = function() {
		StateManager.push(this);
	}
	
	/**
	* hide();
	*  - Use this to hide the state.
	**/
	this.hide = function() {
		if (StateManager.states.length == 0) return;
		if (StateManager.states[StateManager.states.length-1].name == this.name) StateManager.pop();
		else StateManager.removeState(this);
	}
}